//#include <userint.h>

/*= DIGISTANT 14xx ========================================================*/
/* LabWindows Instrument Driver                                            */
/* Original Release: V1.0  13.10.94                                        */
/* By: M.Westermann                                                        */
/* Originally written in C                                                 */
/* Modification History: 	06.05.2003 by Thomas Meder                     */
/*							Version V2.0								   */
/*							Dummy read added in GPIB Function 			   */
/*							"bp14xx_write_msg_gpib" 					   */
/*=========================================================================*/
#include <rs232.h>
#include <formatio.h>
#include <utility.h>
#include <gpib.h>
#include <ansi_c.h>

#include "bp14xx.h"

/*= STATIC VARIABLES ======================================================*/
static int interface;                     /* 0 = GPIB; 1 = Serial*/
static int setup_first ;                  /* remark the first initial*/
static double std_timeout;                /* remark the user timeout*/
static int max_bcc_repeat;                /* Max-Bcc-Repeat */
static int bcc_repeat;                    /* Bcc-Repeat counter*/
static int bcc_state;                     /* Blockcheck state ON or OFF*/
static int instr_cnt;
static int bp14xx_err;

static int far address[bp14xx_MAX_INSTR + 1];  /*GPIB-Adress*/
static int far bd[bp14xx_MAX_INSTR + 1];       /*Gpib Board*/
static char cmd[bp14xx_MAX_CMD+1];
static char buf[bp14xx_MAX_CMD+1];
static char out_buf[bp14xx_MAX_CMD+1];       /* buffer for serial send */
static char in_buf[bp14xx_MAX_CMD+1];        /* buffer for serial receive */
static char swap_buf[bp14xx_MAX_CMD+1];      /* global serial buffer */
static int port_addr[4] ;                    /* RS232-Port Array*/
static int int_level[4] ;                    /* RS232-Interrupt-Level Array*/
static int dev_port[bp14xx_MAX_INSTR];       /* Device Port*/
static int dev_group_addr[bp14xx_MAX_INSTR]; /* Device Group serial Adress*/
static int dev_user_addr[bp14xx_MAX_INSTR];  /* Device User serial Adress */



/*= UTILITY ROUTINES ======================================================*/

int _VI_FUNC  bp14xx_invalid_integer_range (int, int, int, int);
int _VI_FUNC  bp14xx_invalid_real_range (double, double, double, int);
int _VI_FUNC  bp14xx_open_instr_gpib(int );
int _VI_FUNC  bp14xx_device_closed_gpib (int);
int _VI_FUNC  bp14xx_close_instr_gpib (int);
int _VI_FUNC  bp14xx_send (int ,char*);
int _VI_FUNC  bp14xx_receive(int ,char*);
int _VI_FUNC  bp14xx_read_msg_gpib (int,char*,int);
int _VI_FUNC  bp14xx_write_msg_gpib (int,char *, int);
void bp14xx_setup_arrays (void);
int _VI_FUNC  bp14xx_calculate_bcc(char* );
int _VI_FUNC  bp14xx_send_x3_28(int,char*,char*);
int _VI_FUNC  bp14xx_send_x3_28_message(int,char*,char*);
int _VI_FUNC  bp14xx_receive_x3_28(int,char*,char*,char*);
int _VI_FUNC  bp14xx_receive_and_ack(int,char*);
int _VI_FUNC  bp14xx_close_instr_serial (int);
int _VI_FUNC  bp14xx_device_closed_serial (int);
int _VI_FUNC  bp14xx_open_instr_serial (int,int,int,int);


/*=========================================================================*/
/* This function opens the instrument, queries the instrument for its ID,  */
/* and initializes the instrument to a known state.                        */
/*=========================================================================*/
int _VI_FUNC  bp14xx_init_gpib(int addr,int* instrID)
{
    int ID;
    
    bp14xx_err = 0;
    interface = bp14xx_GPIB;            /* GPIB */

    if (bp14xx_invalid_integer_range (addr, 0, 30,  -1) != 0)
        return bp14xx_err;

    ID = bp14xx_open_instr_gpib (addr);
    if (ID <= 0)
      return bp14xx_err;

    /*  terminate read on LF  */
    if (ibeos (bd[ID], 0x40a) < 0)
    {
     bp14xx_err = 237;
     return bp14xx_err;
    }
    *instrID = ID;
    ibconfig (bd[ID],6 ,1);   /* Repeat Addressing */
    
    bp14xx_setup_arrays ();
    
    return(0);
}
/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC  bp14xx_init_serial(int port,int baud_rate,double timeout,int set_bcc_state,
                   int g_addr,int u_addr,int *instrID)
{
    int ID;
    interface = bp14xx_SERIAL;   /* Serial */

    bp14xx_err = 0;
    bp14xx_setup_arrays ();

    if (bp14xx_invalid_integer_range (port, 1,4, -1) != 0)
     return bp14xx_err;
    if (bp14xx_invalid_integer_range (baud_rate, 1200, 19200, -2) != 0)
     return bp14xx_err;
    if (bp14xx_invalid_real_range (timeout, 0.0F, 72000.0F, -3) != 0)
     return bp14xx_err;
    if (bp14xx_invalid_integer_range (set_bcc_state, 0,1, -4) != 0)
     return bp14xx_err;
    if (bp14xx_invalid_integer_range (g_addr, 0,15, -5) != 0)
     return bp14xx_err;
    if (bp14xx_invalid_integer_range (u_addr, 0,15, -6) != 0)
     return bp14xx_err;

    ID = bp14xx_open_instr_serial(port,baud_rate,g_addr,u_addr);
    if(ID <= 0)
     return bp14xx_err;

    SetComTime (port, timeout);
    std_timeout = timeout;

    if (rs232err != 0)
        return (rs232err);

    max_bcc_repeat = 5;       /* default*/
    bcc_state = set_bcc_state;
    
	
	
    

    *instrID = ID;

    return(0);
}
/* ========================================================================= */
/*  This function puts the device into local mode.                              */
/* ========================================================================= */
int _VI_FUNC  bp14xx_local_gpib(int instrID)
{
    bp14xx_err = 0;
    if (bp14xx_invalid_integer_range (instrID, 1, bp14xx_MAX_INSTR,  -1) != 0)
        return bp14xx_err;
    if (bp14xx_device_closed_gpib (instrID) != 0)
        return bp14xx_err;
    if ((ibloc (bd[instrID]) & (int)0x8000 != 0) != 0)
    {
        bp14xx_err = 234;
        return bp14xx_err;
    }
    return bp14xx_err;
}
/* ========================================================================= */
/*  This function change the device timeout                                  */
/* ========================================================================= */
int _VI_FUNC  bp14xx_timout_gpib(int instrID,int value)
{
    bp14xx_err = 0;
    if (bp14xx_invalid_integer_range (instrID, 1, bp14xx_MAX_INSTR,  -1) != 0)
        return bp14xx_err;
        
    if ((ibtmo (bd[instrID], value) != 0) != 0)
    {
        bp14xx_err = 234;
        return bp14xx_err;
    }
    return bp14xx_err;
}
/* ========================================================================= */
/*  This function output the value                                           */
/* ========================================================================= */
int _VI_FUNC  bp14xx_set_value(int instrID,char* value)
{
    bp14xx_err = 0;

    if (bp14xx_invalid_integer_range (instrID, 1, bp14xx_MAX_INSTR,  -1) != 0)
        return bp14xx_err;


    if (bp14xx_send (instrID, value) != 0)
      return bp14xx_err;

    return(0);
}
/* ========================================================================= */
/*  This function return output the value                                    */
/* ========================================================================= */
int _VI_FUNC  bp14xx_get_value(int instrID,char *value)
{
    bp14xx_err = 0;

    if (bp14xx_invalid_integer_range (instrID, 1, bp14xx_MAX_INSTR,  -1) != 0)
        return bp14xx_err;


    if(bp14xx_receive(instrID,buf) != 0)
     return bp14xx_err;

    if (Scan (buf, "%s>%s[w6]", value) != 1)
    {
     bp14xx_err = 236;
     return bp14xx_err;
    }
    return(0);
}
/*=========================================================================*/
/* Function: Serial Poll                                                   */
/* Purpose:  This function performs a serial poll on the instrument.       */
/*           The status byte of the instrument is placed in the response   */
/*           variable. The return value is equal to the global error       */
/*           variable.                                                     */
/*=========================================================================*/
int _VI_FUNC  bp14xx_poll_gpib (int instrID,char *response)
{
    if (ibrsp (bd[instrID], response) <= 0)
        bp14xx_err = 226;
    else
        bp14xx_err = 0;

    return bp14xx_err;
}

/*=========================================================================*/
/* This function Send                                                      */
/*=========================================================================*/
int _VI_FUNC  bp14xx_send(int instrID,char* send_buffer)
{
   bp14xx_err = 0;

   Fmt(cmd,"%s<%s[w6]\r\n",send_buffer);

   if(interface == bp14xx_GPIB)  /* GPIB ?? */
   {
    bp14xx_err = bp14xx_write_msg_gpib(instrID,cmd,NumFmtdBytes ());

    if(bp14xx_err != 0)
     return (bp14xx_err);
   }
   else    /* Serial */
   {
    bp14xx_err = bp14xx_send_x3_28(instrID,"sr",cmd);
    if(bp14xx_err != 0)
     return (bp14xx_err);
   }

   return(0);
}
/*=========================================================================*/
/* This function receive SCPI                                              */
/*=========================================================================*/
int _VI_FUNC  bp14xx_receive(int instrID,char* receive_buffer)
{
   bp14xx_err = 0;


   if(interface == bp14xx_GPIB)  /* GPIB ?? */
   {
/*    bp14xx_err = bp14xx_write_msg_gpib(instrID,cmd,NumFmtdBytes ());
    if(bp14xx_err != 0)
     return (bp14xx_err);*/

    bp14xx_err = bp14xx_read_msg_gpib(instrID,receive_buffer,bp14xx_MAX_CMD);
    if(bp14xx_err != 0)
     return (bp14xx_err);
   }
   else
   {
    bp14xx_err = bp14xx_send_x3_28(instrID,"sr",cmd);
    if(bp14xx_err != 0)
     return (bp14xx_err);

    bp14xx_err = bp14xx_receive_x3_28(instrID,"po","",receive_buffer);
    if(bp14xx_err != 0)
     return (bp14xx_err);
   }
  /* FmtOut("\n**%s**",receive_buffer);*/

   return(0);
}
/* ========================================================================= */
/*  This function closes the instrument.                                     */
/* ========================================================================= */
int _VI_FUNC  bp14xx_close (int instrID)
{
    bp14xx_err = 0;

    if (bp14xx_invalid_integer_range (instrID, 1, bp14xx_MAX_INSTR,  -1) != 0)
     return (bp14xx_err);

    if(interface == bp14xx_GPIB)  /* GPIB ?? */
    {
     if (bp14xx_device_closed_gpib (instrID) != 0)
      return (bp14xx_err);
     bp14xx_close_instr_gpib (instrID);
    }
    else
    {
     if (bp14xx_device_closed_serial (instrID) != 0)
      return (bp14xx_err);
     bp14xx_close_instr_serial (instrID);
    }
    return (bp14xx_err);
}
/* = GPIB UTILITY ROUTINES == ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the GPIB device table for the instrument.  If successful, the        */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp14xx_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC  bp14xx_open_instr_gpib(int addr)
{
    int i;
    int instrID;

    instrID = 0;
    /*  Check to see if the instrument is already in the Instrument Table.  */
    for (i = 1; i <= bp14xx_MAX_INSTR; i++)
        if (address[i] == addr)
        {
            instrID = i;
            i = bp14xx_MAX_INSTR;
        }
    /*  If it is not in the instrument table, open an entry for the instrument.  */
    if (instrID <= 0)
        for (i = 1; i <= bp14xx_MAX_INSTR; i++)
            if (!address[i])
            {
                instrID = i;
                i = bp14xx_MAX_INSTR;
            }
    /*  If an entry could not be opened in the Instrument Table, return an error. */
    if (instrID <= 0)
    {
        bp14xx_err = 220;
        return  -1;
    }
    /*   If the device has not been opened in the GPIB device table (bd[ID] = 0), */
    /*   then open it.                                                            */
    if (bd[instrID] <= 0)
    {
        if (instr_cnt <= 0)
            CloseInstrDevs ("bp14xx");
        bd[instrID] = OpenDev ("", "bp14xx");
        if (bd[instrID] <= 0)
        {
            bp14xx_err = 220;
            return  -1;
        }
    }
    /*   Change the primary address of the device     */
    if (ibpad (bd[instrID], addr) < 0)
    {
        bp14xx_err = 233;
        return  -1;
    }
    address[instrID] = addr;
    instr_cnt++;
    return instrID;
}

/* ========================================================================== */
/*  This function closes the instrument by removing it from the GPIB device   */
/*  table and setting the address and the bd to zero in the Instrument Table. */
/*  The return value is equal to the global error variable.                   */
/* ========================================================================== */
int _VI_FUNC  bp14xx_close_instr_gpib (int instrID)
{

    if (bd[instrID] != 0)  {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt--;
    }
    else
        bp14xx_err = 221;
    return bp14xx_err;
}
/* ========================================================================= */
/*  This function checks to see if the module has been initialized.  If the  */
/*  device has not been opened, a 1 is returned, 0 otherwise.                */
/* ========================================================================= */
int _VI_FUNC  bp14xx_device_closed_gpib (int instrID)
{

    if (bd[instrID] <= 0)
    {
        bp14xx_err = 232;
        return  -1;
    }
    return 0;
}

/* ========================================================================= */
/*  This function reads a buffer of data from the instrument. The return     */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp14xx_read_msg_gpib(int instrID,char *msg,int cnt)
{

    if (ibrd (instrID, msg, (long)cnt) <= 0)
        bp14xx_err = 231;
    else
        bp14xx_err = 0;

	if(ibsta & ERR)
	 bp14xx_err = 231;
     

    return bp14xx_err;
}

/* ========================================================================= */
/*  This function writes a buffer of data to the instrument. The return      */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp14xx_write_msg_gpib (int instrID,char *msg,int cnt)
{
	char buf[10];

	// Dummy Read ausfhren 06.05.2003/MT
    bp14xx_read_msg_gpib (bd[instrID], buf, 10);    

    if (ibwrt (bd[instrID], msg, (long)cnt) <= 0)
        bp14xx_err = 230;
    else
        bp14xx_err = 0;
	if(ibsta & ERR)
	 bp14xx_err = 230;
      
    return bp14xx_err;

}
/* = SERIAL UTILITY ROUTINES ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the device table for the instrument.  If successful, the            */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp14xx_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC  bp14xx_open_instr_serial (int port,int baud_rate,int g_addr,int u_addr)
{
    int i, instrID;

    instrID = 0;

    /*--Check to see if the instrument is already in the Instrument Table.-*/

    for (i=1; i< bp14xx_MAX_INSTR; i++)
    {
/*    FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
     if((dev_group_addr[i] == g_addr) && (dev_user_addr[i] == u_addr))
     {
        instrID = i;
        i = bp14xx_MAX_INSTR;
     }
    }

   /*------ If it is not in the instrument table, open an entry for the
            instrument. ---------------------------------------------------*/

    if (instrID <= 0)
    {
     for (i=1; i< bp14xx_MAX_INSTR; i++)
     {
/*     FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
        if( (dev_group_addr[i] < 0) && (dev_user_addr[i] < 0))
        {
         instrID = i;
         i = bp14xx_MAX_INSTR;
        }
     }
    }

    /*----- If an entry could not be opened in the Instrument Table,
            return an error.-----------------------------------------------*/

    if (instrID <= 0)
    {
     bp14xx_err = 220;
     return -1;
    }


    dev_port[instrID] = port;
        /* CVI Function  */
    OpenComConfig (dev_port[instrID], "", baud_rate,0, 8, 1, 512, 512);
     
	/* Dos
    OpenComConfig (dev_port[instrID], baud_rate,0, 8, 1, 512, 512,
                                      port_addr[port-1],int_level[port-1]);*/

                                      
    if (rs232err != 0)
    {
     dev_port[instrID] = 0;
     dev_group_addr[instrID] = -1;
     dev_user_addr[instrID] = -1;
     bp14xx_err = rs232err;
     return (-1);
    }
    dev_group_addr[instrID] = g_addr;
    dev_user_addr[instrID] = u_addr;
    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           device table and setting the address and the bd to zero       */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int _VI_FUNC  bp14xx_close_instr_serial (int instrID)
{
    if( (dev_group_addr[instrID] >= 0) && (dev_user_addr[instrID] >= 0))
    {
      dev_port[instrID] = 0;
      dev_group_addr[instrID] = -1;
      dev_user_addr[instrID] = -1;
    }
    else
     bp14xx_err = 221;

    return bp14xx_err;
 }
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int _VI_FUNC  bp14xx_device_closed_serial (int instrID)
{
     if( (dev_group_addr[instrID] < 0) && (dev_user_addr[instrID] < 0))
     {
        bp14xx_err = 232;
        return -1;
     }
     return 0;
}

/*=========================================================================*/
/* This function send a comand and message and wait for ACK                */
/*=========================================================================*/
int _VI_FUNC  bp14xx_send_x3_28(int instrID,char* cmd,char* msg)
{
    int ret,r_byte;

    FlushInQ(dev_port[instrID]);           /* Clear Port */

    bcc_repeat = 0;
    for(;;)
    {
     ret = bp14xx_send_x3_28_message(instrID,cmd,msg);
     if(ret != 0)
       return (ret);     /* ERROR*/

     r_byte = ComRdByte(dev_port[instrID]);
     if (rs232err != 0)
      return (231);

    /* FmtOut("\nsend ret %c %d\n",r_byte,bcc_repeat);*/
     Delay(0.01);
     switch(r_byte)
     {
      case ACK:
       return  (0);
       break;
      case NAK:
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
       break;
      case EOT:
       return (301);
       break;
     }
    }

    return (0);
}

/*=========================================================================*/
/* This function send comand and message and receive a string              */
/*=========================================================================*/
int _VI_FUNC  bp14xx_receive_x3_28(int instrID,char* cmd,char* msg,char* receive_buf)
{
    int ret;

    FlushInQ(dev_port[instrID]);           /* Clear Port */
    ret = bp14xx_send_x3_28_message(instrID,cmd,msg);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ret = bp14xx_receive_and_ack(instrID,receive_buf);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ComRdByte(dev_port[instrID]);   /* Read the EOT from Port*/

    return(0);
}
/*=========================================================================*/
/* This function receive a string                                          */
/*=========================================================================*/
int _VI_FUNC  bp14xx_receive_and_ack(int instrID,char* receive_buf)
{
    int r_byte,r_bcc,bcc,l,first_byte;

    bcc_repeat = 0;
    for(;;)
    {
     first_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT or STX from Port*/
     if(first_byte == EOT)
      return (301);

     r_byte = ComRdTerm(dev_port[instrID],in_buf,bp14xx_MAX_CMD-1,ETX);/* Read to ETX*/
     if (rs232err != 0)
      return (231);

     if(bcc_state == 1)
     {
      r_bcc = ComRdByte(dev_port[instrID]);   /* Read the BCC from Port*/
      if (rs232err != 0)
       return (231);
     }
     in_buf[r_byte] = ETX;            /* Calculate bcc with ETX */
     in_buf[r_byte+1] = NUL;          /* Terminate the String*/
     l = StringLength(in_buf);

    /* FmtOut("\nin %c %s",first_byte,in_buf);*/

     CopyString (swap_buf, 0,in_buf, 0, l);   /* calculate bcc without STX */
     if(bcc_state == 1)
     {
      bcc = bp14xx_calculate_bcc(swap_buf);
      if(r_bcc != bcc)
      {
       ComWrtByte(dev_port[instrID],NAK);
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
      }
      else
      {
       ComWrtByte(dev_port[instrID],ACK);
       CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
       return(0);                /* 0 = ACK == OK*/
      }
     }
     else    /* Without bcc check*/
     {
      ComWrtByte(dev_port[instrID],ACK);
      CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
      return(0);                /* 0 = ACK == OK*/
     }
    }

    return(0);
}

/*=========================================================================*/
/* This function send a comand and message                                 */
/*=========================================================================*/
int _VI_FUNC  bp14xx_send_x3_28_message(int instrID,char* cmd,char* msg)
{
    int ret,l;
    int bcc;

    ret = CompareStrings (msg, 0,"", 0, 0); /* Send a command without msg ?*/
    if(ret == 0)
     Fmt(out_buf,"%s<%c%x%x%x%x%s%c", EOT,
                                      dev_group_addr[instrID],
                                      dev_group_addr[instrID],
                                      dev_user_addr[instrID],
                                      dev_user_addr[instrID],
                                      cmd,ENQ); /* without msg*/
    else
    {
     Fmt(swap_buf ,"%s<%s%c",msg,ETX);   /* with msg*/
     if(bcc_state == 1)
     {
      bcc = bp14xx_calculate_bcc(swap_buf);
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s%c", EOT,
                                           dev_group_addr[instrID],
                                           dev_group_addr[instrID],
                                           dev_user_addr[instrID],
                                           dev_user_addr[instrID],
                                           cmd,STX,swap_buf,bcc);
     }
     else
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s", EOT,
                                         dev_group_addr[instrID],
                                         dev_group_addr[instrID],
                                         dev_user_addr[instrID],
                                         dev_user_addr[instrID],
                                         cmd,STX,swap_buf);
    }

   /* FmtOut("out %s",out_buf);*/

    l = StringLength(out_buf);
    ret = ComWrt(dev_port[instrID], out_buf, l);     /* Write the Message */
    if (rs232err != 0)
     return (230);

    return (0);
}
/*=========================================================================*/
/* This function calculate the block check                                 */
/*=========================================================================*/
int _VI_FUNC  bp14xx_calculate_bcc(char* msg)
{
    int l,i,bcc;

    bcc = 0;
    l = StringLength(msg);
    for(i = 0;i <= l;i++)
     bcc ^= msg[i];     /* exclusive OR */

    bcc |= 0x80;    /* OR Bit 8 */

    return (bcc);
}

/* = Global UTILITY ROUTINES ==============================================*/
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp14xx_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp14xx_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp14xx_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp14xx_err = err_code;
    return -1;
  }
  return 0;
}
void bp14xx_setup_arrays (void)
{
    int i;

    if(!setup_first)/*----------- Reset the Arrays before initial --------*/
    {
     for (i=1; i< bp14xx_MAX_INSTR; i++)
     {
       dev_group_addr[i] = -1;
       dev_user_addr[i]  = -1;
       dev_port[i]       =  0;
     }
     setup_first = 1;
    }
    port_addr[0] = 0x3f8;
    port_addr[1] = 0x2f8;
    port_addr[2] = 0x3e8;
    port_addr[3] = 0x2e8;

    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;
}
/*void main(void)
{
 int instID;
 int range;
 double limit;
 char buf[100];
 int sense;
 bp14xx_init_serial (1, 9600, 1.0, 0, 0, 0, buf, 1,&instID );


} */
